import 'package:auto_route/auto_route.dart';
import 'package:auto_size_text/auto_size_text.dart';
import 'package:flutter/material.dart';
import 'package:flutter_remix/flutter_remix.dart';
import 'package:remixicon_updated/remixicon_updated.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';

import '../../../../app_constants.dart';
import 'package:admin_desktop/application/providers.dart';
import '../../../../infrastructure/services/utils.dart';
import 'package:admin_desktop/presentation/components/components.dart';
import 'package:admin_desktop/presentation/components/text_fields/custom_textformfield.dart';
import '../../../app_assets.dart';
import '../../../routes/app_router.dart';
import 'package:admin_desktop/presentation/styles/style.dart';
import '../../main/widgets/language/languages_modal.dart';
import 'widgets/custom_passwords.dart';

@RoutePage()
class LoginPage extends ConsumerStatefulWidget {
  const LoginPage({super.key});

  @override
  ConsumerState<LoginPage> createState() => _LoginPageState();
}

class _LoginPageState extends ConsumerState<LoginPage>
    with SingleTickerProviderStateMixin {
  late TabController _tabController;
  late TextEditingController _phoneController;
  late TextEditingController _emailController;
  late TextEditingController _passwordController;

  @override
  void initState() {
    _phoneController = TextEditingController();
    _emailController = TextEditingController();
    _passwordController = TextEditingController();
    _tabController = TabController(
      initialIndex: AppHelpers.getAuthOption() == SignUpType.email ? 1 : 0,
      length: 2,
      vsync: this,
    );
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback(
      (_) => ref.read(languagesProvider.notifier).checkLanguage(),
    );
  }

  @override
  void dispose() {
    _phoneController.dispose();
    _emailController.dispose();
    _passwordController.dispose();
    _tabController.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    ref.listen(
      languagesProvider,
      (previous, next) {
        if (next.isSelectLanguage &&
            next.isSelectLanguage != previous?.isSelectLanguage) {
          AppHelpers.showAlertDialog(
            context: context,
            child: SizedBox(
              height: MediaQuery.sizeOf(context).height / 2,
              width: MediaQuery.sizeOf(context).width / 4,
              child: LanguagesModal(
                afterUpdate: () {
                  context.maybePop();
                },
              ),
            ),
          );
        }
      },
    );
    final notifier = ref.read(loginProvider.notifier);
    final state = ref.watch(loginProvider);
    return KeyboardDismisser(
      child: AbsorbPointer(
        absorbing: state.isLoading,
        child: Scaffold(
          resizeToAvoidBottomInset: false,
          backgroundColor: Style.mainBack,
          body: SizedBox(
            width: MediaQuery.sizeOf(context).width,
            child: Row(
              children: [
                SafeArea(
                  child: Container(
                    constraints: BoxConstraints(maxWidth: 536.r),
                    child: Padding(
                      padding: EdgeInsets.only(left: 50.r, right: 50.r),
                      child: Column(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          42.verticalSpace,
                          Row(
                            children: [
                              CommonImage(
                                url: AppHelpers.getFavicon(),
                                height: 40,
                                width: 40,
                              ),
                              12.horizontalSpace,
                              Expanded(
                                child: AutoSizeText(
                                  AppHelpers.getAppName(),
                                  style: Style.interBold(size: 28),
                                  maxLines: 1,
                                  minFontSize: 24,
                                ),
                              ),
                            ],
                          ),
                          56.verticalSpace,
                          Text(
                            AppHelpers.getTranslation(TrKeys.login),
                            style: Style.interBold(
                              size: 32,
                              color: Style.black,
                            ),
                          ),
                          36.verticalSpace,
                          if (AppHelpers.getAuthOption() == SignUpType.both)
                            AuthTabBar(
                              tabController: _tabController,
                              tabs: [
                                AuthTab(
                                  text: TrKeys.phone,
                                  icon: FlutterRemix.phone_fill,
                                ),
                                AuthTab(
                                  text: TrKeys.email,
                                  icon: FlutterRemix.mail_fill,
                                ),
                              ],
                            ),
                          16.verticalSpace,
                          SizedBox(
                            height: 76.r,
                            child: TabBarView(
                              controller: _tabController,
                              physics: NeverScrollableScrollPhysics(),
                              children: [
                                CustomPhoneField(
                                  onChanged: (phoneNum) {
                                    _phoneController.text =
                                        phoneNum.completeNumber;
                                    _emailController.clear();
                                  },
                                ),
                                CustomTextField(
                                  hintText: AppHelpers.getTranslation(
                                      TrKeys.typeSomething),
                                  onChanged: notifier.setEmail,
                                  textController: _emailController,
                                  inputType: TextInputType.emailAddress,
                                  textCapitalization: TextCapitalization.none,
                                  isError: state.isLoginError ||
                                      state.isEmailNotValid,
                                  descriptionText: state.isEmailNotValid
                                      ? AppHelpers.getTranslation(
                                          TrKeys.emailIsNotValid)
                                      : (state.isLoginError
                                          ? AppHelpers.getTranslation(TrKeys
                                              .loginCredentialsAreNotValid)
                                          : null),
                                  onFieldSubmitted: (value) => notifier.login(
                                    email: _emailController.text.trim(),
                                    phone: _phoneController.text.trim(),
                                    password: _passwordController.text.trim(),
                                    checkYourNetwork: () {
                                      AppHelpers.showSnackBar(
                                        context,
                                        AppHelpers.getTranslation(
                                            TrKeys.checkYourNetworkConnection),
                                      );
                                    },
                                    unAuthorised: () {
                                      AppHelpers.showSnackBar(
                                        context,
                                        AppHelpers.getTranslation(
                                            TrKeys.emailNotVerifiedYet),
                                      );
                                    },
                                    goToMain: () {
                                      context.replaceRoute(const MainRoute());
                                    },
                                  ),
                                ),
                              ],
                            ),
                          ),
                          16.verticalSpace,
                          Text(
                            AppHelpers.getTranslation(TrKeys.password),
                            style: Style.interNormal(
                              size: 14,
                            ),
                          ),
                          CustomTextField(
                            textController: _passwordController,
                            hintText:
                                AppHelpers.getTranslation(TrKeys.typeSomething),
                            obscure: state.showPassword,
                            onChanged: notifier.setPassword,
                            textCapitalization: TextCapitalization.none,
                            isError:
                                state.isLoginError || state.isPasswordNotValid,
                            descriptionText: state.isPasswordNotValid
                                ? AppHelpers.getTranslation(TrKeys
                                    .passwordShouldContainMinimum8Characters)
                                : (state.isLoginError
                                    ? AppHelpers.getTranslation(
                                        TrKeys.loginCredentialsAreNotValid)
                                    : null),
                            suffixIcon: IconButton(
                              splashRadius: 25.r,
                              icon: Icon(
                                state.showPassword
                                    ? Remix.eye_line
                                    : Remix.eye_close_line,
                                color: Style.black,
                                size: 20.r,
                              ),
                              onPressed: () =>
                                  notifier.setShowPassword(!state.showPassword),
                            ),
                            onFieldSubmitted: (value) => notifier.login(
                              email: _emailController.text.trim(),
                              phone: _phoneController.text.trim(),
                              password: _passwordController.text.trim(),
                              checkYourNetwork: () {
                                AppHelpers.showSnackBar(
                                  context,
                                  AppHelpers.getTranslation(
                                      TrKeys.checkYourNetworkConnection),
                                );
                              },
                              unAuthorised: () {
                                AppHelpers.showSnackBar(
                                  context,
                                  AppHelpers.getTranslation(
                                      TrKeys.emailNotVerifiedYet),
                                );
                              },
                              goToMain: () {
                                bool checkPin =
                                    LocalStorage.getPinCode().isEmpty;
                                context.replaceRoute(
                                    PinCodeRoute(isNewPassword: checkPin));
                              },
                            ),
                          ),
                          42.verticalSpace,
                          Row(
                            children: [
                              CustomCheckbox(isActive: true, onTap: () {}),
                              14.horizontalSpace,
                              Text(
                                AppHelpers.getTranslation(TrKeys.keepMe),
                                style: Style.interNormal(
                                  size: 14,
                                ),
                              ),
                            ],
                          ),
                          56.verticalSpace,
                          LoginButton(
                            isLoading: state.isLoading,
                            title: AppHelpers.getTranslation(TrKeys.login),
                            onPressed: () {
                              notifier
                                  .login(
                                email: _emailController.text.trim(),
                                phone: _phoneController.text.trim(),
                                password: _passwordController.text.trim(),
                                checkYourNetwork: () {
                                  AppHelpers.showSnackBar(
                                    context,
                                    AppHelpers.getTranslation(
                                        TrKeys.checkYourNetworkConnection),
                                  );
                                },
                                unAuthorised: () {
                                  AppHelpers.showSnackBar(
                                    context,
                                    AppHelpers.getTranslation(
                                        TrKeys.emailNotVerifiedYet),
                                  );
                                },
                                goToMain: () {
                                  context.replaceRoute(
                                      PinCodeRoute(isNewPassword: true));
                                },
                              )
                                  .then((s) {
                                if (AppHelpers.getUserRole != TrKeys.master) {
                                  ref.read(mainProvider.notifier).fetchMyShop();
                                }
                              });
                            },
                          ),
                          36.verticalSpace,
                          if (AppConstants.isDemo)
                            Column(
                              children: [
                                CustomPasswords(
                                  type: TrKeys.seller,
                                  onTap: () {
                                    _tabController.animateTo(1,
                                        duration: Duration(microseconds: 300));
                                    _emailController.text =
                                        AppConstants.demoSellerLogin;
                                    _passwordController.text =
                                        AppConstants.demoSellerPassword;
                                    _phoneController.clear();
                                    notifier
                                        .setEmail(AppConstants.demoSellerLogin);
                                    notifier.setPassword(
                                        AppConstants.demoSellerPassword);
                                  },
                                ),
                                24.verticalSpace,
                                CustomPasswords(
                                  type: TrKeys.master,
                                  onTap: () {
                                    _tabController.animateTo(1,
                                        duration: Duration(microseconds: 300));
                                    _emailController.text =
                                        AppConstants.demoSellerLogin2;
                                    _passwordController.text =
                                        AppConstants.demoSellerPassword2;
                                    _phoneController.clear();
                                    notifier.setEmail(
                                        AppConstants.demoSellerLogin2);
                                    notifier.setPassword(
                                        AppConstants.demoSellerPassword2);
                                  },
                                ),
                              ],
                            ),
                          const Spacer(),
                        ],
                      ),
                    ),
                  ),
                ),
                Expanded(
                  child: Image.asset(
                    Assets.pngSplash,
                    height: double.infinity,
                    fit: BoxFit.cover,
                  ),
                ),
              ],
            ),
          ),
        ),
      ),
    );
  }
}
